<?php
/**
* contains class {@link Contact}
* @package backEnd
* @author Tobias Schlatter
*/

/** */
require_once('constants.inc');
require_once('lib/utilities/StringHelper.class.php');
require_once('lib/utilities/EmailHelper.class.php');
require_once('lib/backEnd/DB.class.php');
require_once('lib/backEnd/ErrorHandler.class.php');
require_once('lib/backEnd/PluginManager.class.php');
require_once('lib/utilities/RightsManager.class.php');

/**
* Represents a contact
*
* Handles all database-communication, in order to retrieve and save contact-data
* @package backEnd
*/
class Contact
{
    /**
    * @var array comes from database, contains 1 row from contact table
    */
	var $contact;
    
    /**
    * @var array used to cache value groups
    */
    var $valueGroups;
    
    /**
    * @var integer used to cache, if the contact has a user attached
    */
    var $user;

    /**
    * Constructor; initializes class
    *
    * If {@link $val} is an array, it is used as the data array,
    * if {@link $val} isn't defined, a new contact is created
    * else the class with id {@link $val} is loaded from database
    * @param integer|array $val Id of contact, or array with data for contact
    */
    function Contact($val = false)
    {
        
        $this->user = null;
        
        if (is_array($val))
            $this->contact = $val;
        elseif ($val === false)
            $this->contact = array();
        else
            $this->load($val);
    }
    
    /**
    * Loads the contact with id {@link $id} from databse
    * @param $id Id of contact
    * @global ErrorHandler used to report errors
    * @global DB used to connect to the database
    */
    function load($id)
    {
        global $errorHandler,$db;
        
        if(!is_numeric($id))
            $errorHandler->error('invArg','$id is not numeric');
        
        $db->query('SELECT * FROM ' . TABLE_CONTACT . ' AS contact WHERE contact.id = ' . $db->escape($id));
        
        if ($db->rowsAffected() != 1)
            $errorHandler->error('invArg',"The id $id does not exist.");
        
		$this->contact = $db->next();
    }
    
    /**
    * Saves this contact in the database, creates a new contact, if {@link $contact->contact['id']} is not set
    *
    * This function uses the mysql command SHOW COLUMNS, and then inserts all
    * data from {@link $contact->contact}, which has keys that are in the columns
    * of the contact table
    * @global DB used to connect to the database
    * @global ErrorHandler used to report errors
    * @global PluginManager used to execute plugin-hooks
    * @global integer config: how many days is a contact still marked as added upon change, after addition
    */
    function save()
    {
        
        global $db,$errorHandler,$pluginManager, $CONFIG_ADDED_CHANGED_HYSTERESIS;
        
        if (!trim($this->contact['lastname'])) {
            $errorHandler->error('formVal','Please enter the last name');
            return false;
        }

        $privateOK = RightsManager::mayViewPrivateInfo($_SESSION['user'],$this);
        
        $adding = !isset($this->contact['id']) || !$this->contact['id'];

        $pluginManager->changedContactRecord($this,'will_' . ($adding?'add':'change'));
        
        if ($adding) { 
            $this->contact['lastModification'] = 'added';
            $this->contact['whoAdded'] = $_SESSION['user']->id;
        } else { // avoid changing added -> changed too soon
            if(!isset($CONFIG_ADDED_CHANGED_HYSTERESIS))
                $CONFIG_ADDED_CHANGED_HYSTERESIS = 31;
            
            if($this->contact['lastModification'] == 'added') // only for added! deleted -> changed must be instantly possible
            {
                $db->query('SELECT TO_DAYS(NOW()) >= ' . $db->escape($CONFIG_ADDED_CHANGED_HYSTERESIS) . ' +
                            TO_DAYS(' . $db->escape($this->contact['lastUpdate'])  . ') AS ok');
                $r = $db->next();
                if ($r['ok'])
                    $this->contact['lastModification'] = 'changed';
            }
            else
                $this->contact['lastModification'] = 'changed';
        }
            
        $this->contact['whoModified'] = $_SESSION['user']->id; // no hysteresis needed
        $this->contact['lastUpdate'] = date('Y-m-d H:i:s');
        
        $db->query('SHOW COLUMNS FROM ' . TABLE_CONTACT);
        
        $queryHead = 'REPLACE INTO ' . TABLE_CONTACT . ' (';
        $queryBody = 'VALUES (';
        
        while ($r = $db->next())
            if (isset($this->contact[$r['Field']])) {
                $queryHead .= $r['Field'] . ',';
                $queryBody .= $db->escape($this->contact[$r['Field']]) . ',';
            }
        
        $db->free();
        
        $queryHead = mb_substr($queryHead,0,-1) . ') ';
        $queryBody = mb_substr($queryBody,0,-1) . ')';

        $db->query($queryHead . $queryBody);
        
        if ($adding) {
            $this->contact['id'] = $db->insertID();
            if (!$this->contact['id'])
                $errorHandler->error('retVal','insertID() did not return a proper id');
        }
        
        foreach ($this->valueGroups as $k => $v) {
            
            $table = '';
            
            if ($k == 'groups') {
                
                $db->query('DELETE FROM ' . TABLE_GROUPS . ' WHERE id = ' . $db->escape($this->contact['id']));
                
                if (count($v) <= 0)
                    continue;
                
                $gQuery = 'REPLACE INTO ' . TABLE_GROUPS . ' (id,groupid) VALUES ';
                
                foreach ($v as $g) {
                    $db->query('SELECT * FROM ' . TABLE_GROUPLIST . ' WHERE groupname = ' . $db->escape($g['groupname']));
                    if ($r = $db->next())
                        $g['groupid'] = $r['groupid'];
                    else {
                        $db->query('INSERT INTO ' . TABLE_GROUPLIST . ' (groupname) VALUES (' . $db->escape($g['groupname']) . ')');
                        $g['groupid'] = $db->insertID();
                    }
                    
                    $gQuery .= '(' . $db->escape($this->contact['id']) . ',' . $db->escape($g['groupid']) . '),';
                    
                }
                
                $db->query(mb_substr($gQuery,0,-1));
                // Remove overhead
                $db->query('OPTIMIZE TABLE ' . TABLE_GROUPS);
                
                continue;
            }
            
            if ($k == 'addresses') {
                
                $db->query('SHOW COLUMNS FROM ' . TABLE_ADDRESS);
                
                $cols = array();
            
                while ($r = $db->next())
                    $cols[] = $r['Field'];
    
                $db->free();
                
                $prim = -1;
                
                foreach ($v as $vk => $vv) {
                    $tmp = '(';
                    $empty = true;
                    foreach ($cols as $c) {
                        if(!isset($vv[$c])) // skip entries not in input (e.g. latitude, longitude)
                            $vv[$c]='';
                            
                        if ($c == 'id')
                            $tmp .= $db->escape($this->contact['id']) . ',';
                        elseif ($c == 'refid') {
                            if (isset($vv[$c])) {
                                $tmp .= $db->escape($vv[$c]) . ',';
                                $new = false;
                            } else {
                                $tmp .= "'',";
                                $new = true;
                            }
                        } else {
                            $tmp .= $db->escape($vv[$c]) . ',';
                            if (trim($vv[$c]) && $c != 'country')
                                $empty = false;
                        }
                    }
                    
                    if ($empty && !$new)
                        $db->query('DELETE FROM ' . TABLE_ADDRESS . ' WHERE refid = ' . $db->escape($vv['refid']));
                    elseif (!$empty) {
                        $db->query('REPLACE INTO ' . TABLE_ADDRESS . ' VALUES ' . mb_substr($tmp,0,-1) . ')');
                        if (isset($vv['primary']) && $vv['primary']) {
                            $id = $db->insertID();
                            $db->query('UPDATE ' . TABLE_CONTACT . ' SET primaryAddress = ' . $db->escape($id) . ' WHERE id = ' . $db->escape($this->contact['id']));
                        }
                    }
                }
                
                continue;
                
            }
            
            if ($k == 'date') {
                $tbl = TABLE_DATES;
                // Remove old entries
                $db->query('DELETE FROM ' . $tbl . '
                    WHERE (visibility = ' . $db->escape('visible') . '
                        OR visibility = ' . $db->escape('hidden') . ' AND ' . $db->escape($privateOK) . '
                        OR visibility = ' . $db->escape('admin-hidden') . ' AND ' . $db->escape($_SESSION['user']->isAtLeast('admin')) . ')
                    AND id = ' . $db->escape($this->contact['id']));
            } else {
                $tbl = TABLE_PROPERTIES;
                // Remove old entries
                $db->query('DELETE FROM ' . $tbl . '
                    WHERE type = ' . $db->escape($k) . '
                    AND (visibility = ' . $db->escape('visible') . '
                        OR visibility = ' . $db->escape('hidden') . ' AND ' . $db->escape($privateOK) . '
                        OR visibility = ' . $db->escape('admin-hidden') . ' AND ' . $db->escape($_SESSION['user']->isAtLeast('admin')) . ')
                    AND id = ' . $db->escape($this->contact['id']));
            }
            
            
            
            if (count($v) <= 0)
                continue;
                
            $db->query('SHOW COLUMNS FROM ' . $tbl);
            
            $cols = array();
            
            while ($r = $db->next())
                $cols[] = $r['Field'];
            
            $queryContent = 'VALUES ';

            $db->free();
            
            foreach ($v as $vk => $vv) {
                $queryContent .= '(';
                foreach ($cols as $c) {
                    if ($c == 'id')
                        $queryContent .= $db->escape($this->contact['id']) . ',';
                    elseif ($c == 'type')
                        $queryContent .= $db->escape($k) . ',';
                    else
                        $queryContent .= $db->escape($vv[$c]) . ',';
                }
                $queryContent = mb_substr($queryContent,0,-1) . '),';
            }
            
            $queryContent = mb_substr($queryContent,0,-1);
            
            $db->query('INSERT INTO ' . $tbl . ' (' . implode(',',$cols) . ') ' . $queryContent);

            // Remove overhead
            $db->query('OPTIMIZE TABLE ' . $tbl);
        }
        
        $pluginManager->changedContactRecord($this,($adding?'added':'changed'));
        
        $this->load($this->contact['id']);
        
        return true;
        
    }
    
    /**
    * Trash the contact (mark as deleted)
    * @global DB used for database connection
    * @global PluginManager used to execute hooks
    * @global ErrorHandler used to handle errors
    * @return boolean true on success
    */
    function trash() {
        
        global $db, $pluginManager, $errorHandler;
        
        if (!isset($this->contact['id']) || !$this->contact['id'])
            $errorHandler('invArg','This contact does not exist in the database');
        
        $db->query('UPDATE ' . TABLE_CONTACT . ' SET
            whoModified = ' . $db->escape($_SESSION['user']->id) . ',
            lastUpdate = NOW(),
            hidden = 1,
            lastModification = ' . $db->escape('deleted') . '
			WHERE id=' . $db->escape($this->contact['id']) . ' LIMIT 1');

        $pluginManager->changedContactRecord($this,'trashed');
        
        $this->load($this->contact['id']);
        
        return true;
        
    }
    
    /**
    * Delete the contact (really delete)
    * @global DB used for database connection
    * @global PluginManager used to execute hooks
    * @global ErrorHandler used to handle errors
    * @return boolean true on success
    */
    function delete() {
        
        global $db, $pluginManager, $errorHandler;
        
        if (!isset($this->contact['id']) || !$this->contact['id'])
            $errorHandler('invArg','This contact does not exist in the database');

        $pluginManager->changedContactRecord($this,'deleted');
            
        $db->query('SELECT * FROM ' . TABLE_USERS . ' WHERE id = ' . $db->escape($this->contact['id']));
        if ($db->rowsAffected()>0) {
            $r = $db->next();
            $pluginManager->changedUserRecord(new User($r),'deleted');
        }
        
        $tables = array(
            TABLE_CONTACT,
            TABLE_ADDRESS,
            TABLE_PROPERTIES,
            TABLE_GROUPS,
            TABLE_USERS,
            TABLE_DATES
        );
        
        foreach ($tables as $t)
            $db->query('DELETE FROM ' . $t . ' WHERE id = ' . $db->escape($this->contact['id']));
        
        return true;
        
    }
    
    /**
    * Check if there is a user, associated with this contact
    * @return integer user id, if there is no such user: 0
    * @global DB used for database connection
    */
    function isUser() {
        
        if ($this->user !== null)
            return $this->user;
        
        global $db;
        
        if (isset($this->contact['id'])) {
            $db->query('SELECT * FROM ' . TABLE_USERS . ' WHERE id = ' . $db->escape($this->contact['id']));
            if ($db->rowsAffected() > 0) {
               $r = $db->next();
               return ($this->user = intval($r['userid']));
            }
        }

        return ($this->user = 0);
        
    }
    
    /**
    * Gets a value group
    * 
    * A value group is an array, containing associative arrays with column names of
    * the database.
    * @param string $type which value group to return
    * @return array value group
    * @global DB used for database connection
    */
    function getValueGroup($type)
    {
        
        global $db;
        
        if (isset($this->valueGroups[$type]))
            return $this->valueGroups[$type];
        
        $valueGroup = array();
        
        $id = isset($this->contact['id']) ? $db->escape($this->contact['id']) : 0;
        $admin = intval($_SESSION['user']->isAtLeast('admin'));
        $privateOK = RightsManager::mayViewPrivateInfo($_SESSION['user'],$this);
        
        // I know that the joins are not necessary and duplicate data, but it is practical 
        // to have everything at hand in a value group, especially when processing externally
        switch($type)
        {
            case 'addresses':
                $sel = "SELECT * FROM " . TABLE_CONTACT . " AS contact, " . TABLE_ADDRESS . " AS phones ";
                $where = "WHERE contact.id=$id AND contact.id=phones.id AND (contact.hidden = 0 OR $admin)";
                break;
            case 'groups':
                $sel = "SELECT grouplist.groupid, groupname FROM " . TABLE_GROUPS . " AS groups LEFT JOIN " . TABLE_GROUPLIST . " AS grouplist ON groups.groupid=grouplist.groupid";
                $where = "WHERE id=$id";
                break;
            case 'date':
                $sel = "SELECT * FROM " . TABLE_CONTACT . " AS contact, " . TABLE_DATES . " AS dates ";
                $where = "WHERE contact.id=$id
                    AND contact.id=dates.id
                    AND (contact.hidden = 0 OR $admin)
                    AND (visibility = " . $db->escape('visible'). " OR " . $db->escape($privateOK) . ") ORDER BY dates.value";
                break;
            default:
                $sel = "SELECT * FROM " . TABLE_CONTACT . " AS contact, " . TABLE_PROPERTIES . " AS props ";
                $where = "WHERE contact.id=$id
                    AND contact.id=props.id
                    AND (contact.hidden = 0 OR $admin)
                    AND props.type = " . $db->escape($type) . "
                    AND (visibility = " . $db->escape('visible'). " OR " . $db->escape($privateOK) . ") ORDER BY props.type";
        }
        
        // return a 2D array
        //echo "$sel $where";
        $db->query("$sel $where");
        while($row = $db->next())
            $valueGroup[]=$row;
        
        $this->valueGroups[$type] = $valueGroup;
        
        return $valueGroup;
    }
    
    /**
    * Sets a value group
    *
    * Saves the value group in the cache of the contact class. It is written in
    * the database, when the {@link save()} method is called
    * @param string $type which value group to save
    * @param array $valueGroup the value group to save
    * @return boolean true on success
    **/
    function setValueGroup($type,$valueGroup)
    {
        $this->valueGroups[$type] = $valueGroup;
        return true;
    }
    
    /**
    * Gets a contact, which has the e-mail {@link $email}
    * @static
    * @param string $email the e-mail to search for
    * @return array an associative array of the contact row
    * @global DB used for database connection
    */
    function getContactFromEmail($email) {
    
        global $db;
        
        $db->query('SELECT * FROM ' . TABLE_PROPERTIES . ' AS properties, ' . TABLE_CONTACT . ' AS contact
            WHERE properties.id = contact.id AND properties.type = ' . $db->escape('email') . ' AND properties.value = ' . $db->escape($email));
            
        return $db->next();
    
    }
    
    //==============================================
    //===== HTML Output Section               ======
    //==============================================
    
    /**
    * Output the phones of the contact in nice html
    *
    * @param array $vg_x the value group to use, if not set, the one of the contact is used
    * @return string html
    */
    function phones($vg_x = NULL)
    {
        $cont = '';
        
        if($vg_x == NULL)
            $vg_x = $this->getValueGroup('phone');
            
        if (count($vg_x) > 0) {
            
            $cont = '<div class="phones-title">Other Phone Numbers</div><ul class="phones">';
            
            foreach($vg_x as $p)
                $cont .= '<li' . ($p['visibility'] != 'visible'?' class="hidden"':'') . '><span class="phones-label">' . $p['label'] . ':</span> ' .
                    '<span class="phones-info">' . $p['value'] . '</span></li>';
            
            $cont .= '</ul>';
            
        }
        
        return $cont;
        
    }

    /**
    * Output the messaging handles of the contact in nice html
    *
    * @param array $vg_x the value group to use, if not set, the one of the contact is used
    * @return string html
    */
    function messaging($vg_x = NULL)
    {
        $cont = '';
        
        if($vg_x == NULL)
            $vg_x = $this->getValueGroup('chat');
        
        if (count($vg_x) > 0) {
            
            $cont = '<div class="messaging-title">Messaging</div><ul class="messaging">';
            
            foreach($vg_x as $x)
                $cont .= '<li' . ($x['visibility'] != 'visible'?' class="hidden"':'') . '><span class="messaging-label">' . $x['label'] . ':</span> ' .
                    '<span class="messaging-info">' . $x['value'] . '</span></li>';
                    
            $cont .= '</ul>';
            
        }

        return $cont;

    }

    /**
    * Output the homepages of the contact in nice html
    *
    * @param array $vg_x the value group to use, if not set, the one of the contact is used
    * @return string html
    */
    function webs($vg_x = NULL)
    {
        if($vg_x == NULL)
            $vg_x = $this->getValueGroup('www');

        global $lang;
        
        $cont = '';
        foreach($vg_x as $x)
        {
            if(!$cont)
                $cont = '<div class="webs"><span class="webs-title">Websites</span>';
            else
                $cont .= '<div class="webs' . ($x['visibility'] != 'visible'?'-hidden':'') . '"><span class="webs-title">&nbsp;</span>';
            
            $label = $x['label'];
            if(empty($label)) $label = $x['value'];
			$cont .= '<span class="webs-info"><a class="webs" href="' . $x['value'] . '">' . $label . '</a></span></div>';
		}
        
        return $cont;
        
    }

    /**
    * Output the additional information of the contact in nice html
    *
    * @param array $vg_x the value group to use, if not set, the one of the contact is used
    * @return string html
    */
    function additionals($vg_x = NULL)
    {
        $cont = '';
        
        if($vg_x == NULL)
            $vg_x = $this->getValueGroup('other');

        foreach($vg_x as $x)
        {
            if(StringHelper::isHTTP($x['value']))
                $val = '<a href="' . $x['value'] . '">' . $x['value'] . "</a>";
            else
                $val = $x['value'];
			$cont .= '<div class="other' . ($x['visibility'] != 'visible'?'-hidden':'') . '"><span class="other-label">' .$x['label'] . '</span><span class="other-info">' . $val . '</span></div>';
		}
        return $cont;
    }

    /**
    * Return first plain email address to send an email to the contact from TAB
    *
    * @return string plain email address to send an email to the contact from TAB
    */
    function getFirstEmail()
    {
        $vg_x = $this->getValueGroup('email');
        return $vg_x[0]['value'];
    }
    
    /**
    * sends a generic e-mail to the Contact's first email
    *
    * @param string $subject address to send e-mail to
    * @param string $message e-mail message
    * @global CONFIG_PHPMAILER array used to configure phpmailer
    * @global options
    * @global errorHandler    
    */
    
    function sendEMail($subject,$message) {
    
        global $CONFIG_TAB_ROOT, $CONFIG_PHPMAILER, $options, $errorHandler;
        
        require_once("lib/phpmailer/class.phpmailer.php");
        
        $mailer = new PHPMailer();
        
        if(isset($CONFIG_PHPMAILER))
        {
            $mailer->Mailer   = $CONFIG_PHPMAILER['Mailer']; 
            $mailer->Sendmail = $CONFIG_PHPMAILER['Sendmail'];
            $mailer->Host     = $CONFIG_PHPMAILER['Host'];
            $mailer->Port     = $CONFIG_PHPMAILER['Port'];
            $mailer->SMTPAuth = $CONFIG_PHPMAILER['SMTPAuth'];
            $mailer->Username = $CONFIG_PHPMAILER['Username'];         
            $mailer->Password = $CONFIG_PHPMAILER['Password'];
        }
        
        $mailer->From = 'noreply@' . $_SERVER['SERVER_NAME'];
        $mailer->FromName = 'noreply@' . $_SERVER['SERVER_NAME'];
        $mailer->AddAddress($this->getFirstEmail());
        
        $mailer->Subject = $options->getOption('adminEmailSubject') . ' - ' . $subject;
        $mailer->Body    = 'This is an auto-generated message from The Address Book Reloaded at ' . $_SERVER['SERVER_NAME'] .
            ".\n\n" . $message . "\n\n" . $options->getOption('adminEmailFooter');
        
        if(!$mailer->Send())
            $errorHandler->error('mail','Email sending failed: ' . $mail->ErrorInfo,get_class($this));
        else
            $errorHandler->error('ok','Mail sent!');
    }

    /**
    * Output the e-mails of the contact in nice html
    *
    * @param array $vg_x the value group to use, if not set, the one of the contact is used
    * @return string html
    */
    function emails($vg_x = NULL)
    {
        $cont = '';
        
        if($vg_x == NULL)
            $vg_x = $this->getValueGroup('email');
        
        if (count($vg_x) > 0) {
        
            $cont = '<div class="emails-title">E-Mail</div><ul class="emails">';
            
            foreach($vg_x as $m)
                    $cont .= '<li' . ($m['visibility'] != 'visible'?' class="hidden"':'') . '>' . $this->generateEmail($m) . '</li>';
            
            $cont .= '</ul>';
            
        }
    
        return $cont;
	    
    }
    
    /**
    * Generate an e-mail link from one entry of an e-mail value group
    *
    * @param array $m the entry of the value group to use
    * @return string html
    * @static
    */
    function generateEmail($m) {
        return EmailHelper::sendEmailLink($this->contact['firstname'] . ' ' . $this->contact['lastname'] . ($m['label']?' ('.$m['label'].')':'') . ' <' . $m['value'] . '>',
            $m['value']) . 
            ($m['label']?' ('.$m['label'].')':'');
    }
    
    /**
    * Output the addresses of the contact in nice html
    *
    * @param array $vg_x the value group to use, if not set, the one of the contact is used
    * @return string html
    * @global PluginManager used to call hooks
    * @global AddressFromatter used to format the addresses, depending on the countries they are in
    */
    function addresses($vg_x = NULL)
    {
        if($vg_x == NULL)
            $vg_x = $this->getValueGroup('addresses');

        global $pluginManager, $addressFormatter;

        $cont = '<ul class="card-addresses">';
        
        foreach($vg_x as $a) {
            $cont .= '<li>';
            $cont .= '<div class="address-head-line"><span class="address-title">' .
                ($this->contact['primaryAddress'] == $a['refid']?'Primary Address':'Address');
            if ($a['type'])
                $cont .= ' <span class="address-label">(' . $a['type'] . ')</span>';
            $cont .= '</span></div>';
            $cont .= $addressFormatter->formatAddress($a);
            if ($a['phone1'])
                $cont .= '<div class="address-line"><span class="address-phone">' . $a['phone1'] . '</span></div>';
            if ($a['phone2'])
                $cont .= '<div class="address-line"><span class="address-phone">' . $a['phone2'] . '</span></div>';
            
            // handle plugins
            $nav = new Navigation();
            $cont .= $pluginManager->addressOutput($this,$a['refid'],$nav);
            $pluginManager->addressMenu($this,$a['refid'],$nav);
            $cont .= '<div class="address-line">' . $nav->create('address-menu') . '</div>';
            
            
            $cont .= '</li>';
        }
        
        $cont .= '</ul>';
        
        return $cont;
    }

    /**
    * Output the notes of the contact in nice html
    *
    * @return string html
    */
    function notes()
    {
        $notes = $this->contact['notes'];
        
        if(!$notes)
            return '';
            
        $cont = '<div class="notes-title">Notes</div><div class="notes">';
		$cont .= $notes;
		$cont .= '</div>';
        
        return $cont;
        
    }

    /**
    * Output the dates of the contact in nice html
    *
    * @param array $vg_x the value group to use, if not set, the one of the contact is used
    * @return string html
    */
    function dates($vg_x = NULL)
    {
        $cont='';
        
        if($vg_x == NULL)
            $vg_x = $this->getValueGroup('date');
        
        foreach($vg_x as $x)
			$cont .= '<div class="other' . ($x['visibility'] != 'visible'?'-hidden':'') . '"><span class="other-label">' .$x['label'] . '</span><span class="other-info">' . $x['value'] . '</span></div>';
        
        return $cont;        
    }
    
    /**
    * Output the groups of the contact in nice html
    *
    * @param array $vg_x the value group to use, if not set, the one of the contact is used
    * @return string html
    */
    function groups($vg_x = NULL)
    {
        if($vg_x == NULL)
            $vg_x = $this->getValueGroup('groups');
        
        $cont = '<ul class="groups">';
            
        foreach($vg_x as $g)
            $cont .= "<li><a href=\"list.php?group={$g['groupname']}\">{$g['groupname']}</a></li>";
            
        $cont .= '</ul>';
        
        return $cont;
        
    }
    
    /**
    * Output the names (firstname, lastname, nickname, middlename) of the contact in nice html
    *
    * @return string html
    */
    function names()
    {
        
        $cont = "<div class=\"names" . ($this->contact['hidden']?'-hidden':'') . "\">{$this->contact['lastname']}". (!empty($this->contact['firstname']) ? ', ' . $this->contact['firstname'] : '') ." {$this->contact['middlename']}";
        if ($this->contact['nickname'])
            $cont .= " &quot;{$this->contact['nickname']}&quot;";
        if ($this->contact['hidden'])
            $cont .= ' [!]';
        $cont .=  "</div>";
        
        return $cont;
     
    }
    
    // === private section ===
    
    /**
    * Used by serialize, when serializing class
    *
    * @return array list of variables that should be saved
    */
    function __sleep() {
        
        $this->valueGroups = null;
        $id = $this->contact['id'];
        $this->contact = array();
        $this->contact['id'] = $id;
        
        return array('contact');
        
    }
    
    /**
    * Used by serialize, when deserializing class
    */
    function __wakeup() {
        
        $this->load($this->contact['id']);
        
    }
    
}

?>
