<?php

/**
* contains class {@link Image}
* @author Tobias Schlatter
* @package frontEnd
*/

/** */

require_once('lib/utilities/StringHelper.class.php');
require_once('lib/backEnd/Options.class.php');
require_once('lib/backEnd/ErrorHandler.class.php');

/**
* represents the mugshot of a certain contact
* is also capable of resampling/croping image files to a defined size
* @package frontEnd
*/

class Image
{
    /**
    * @var Contact the contact, which the image belongs to
    */
    var $contact;
    
    /**
    * Constructor
    * 
    * Initializes {@link $contact}
    * @param Contact $contact contact which the image should belong to
    */
    function Image(&$contact) {
        $this->contact = &$contact;
    }
    
    /**
    * Resamples/crops image file
    *
    * This method reads the requested file, checks, whether it should
    * just resample it (make the file at most as big as the size defined in
    * the options) or if it should crop it in a way that it will fill the whole
    * space available
    * @param string $file path of the file to resample/crop
    * @return string jpeg image stream
    * @global ErrorHandler used for error handling
    * @global Options used to determine mode and dimensions
    */
    function resampleAndReturn($file) {
        
        global $errorHandler, $options;
        
        $ret = getimagesize($file);
        
        if (!$ret) {
            $errorHandler->error('image','The file seems not to be an image');
            return false;
        }
        
        $width = $ret[0];
        $height = $ret[1];
        
        $maxWidth = $options->getOption('picWidth');
        $maxHeight = $options->getOption('picHeight');
        
        if ($ret[2] == 2 &&
            $width <= $maxWidth &&
            $height <= $maxHeight) // Is jpg image and correct size??
            
            // Keep it as it is
            $imageData = file_get_contents($file);
            
        else {
                
            if (!function_exists('imagejpeg') || !function_exists('imagecreatetruecolor') || !function_exists('imagecopyresampled')) {
                $errorHandler->error('image','Resizing not possible, because there is no gd-lib >= 2.1.0');
                return false;
            }
            
            // 1 = GIF, 2 = JPG, 3 = PNG
            
            switch ($ret[2]) {
                case 1:
                    if (!function_exists('imagecreatefromgif')) {
                        $errorHandler->error('image','GIF-images aren\'t supported');
                        return false;
                    }
                    $img = @imagecreatefromgif($file);
                break;
                case 2:
                    if (!function_exists('imagecreatefromjpeg')) {
                        $errorHandler->error('image','JPEG-images aren\'t supported');
                        return false;
                    }
                    $img = @imagecreatefromjpeg($file);
                break;
                case 3:
                    if (!function_exists('imagecreatefrompng')) {
                        $errorHandler->error('image','PNG-images aren\'t supported');
                        return false;
                    }
                    $img = @imagecreatefrompng($file);
                break;
            }
            
            if (!$img) {
                $errorHandler->error('image','An error occured while loading the image');
                return false;
            }
            
            if ($width <= $maxWidth && $height <= $maxHeight) {
                $newImg = &$img;
            } elseif ($options->getOption('picCrop')) {
                
                if ($width / $maxWidth > $height / $maxHeight) {
                    $yOffset = 0;
                    $xOffset = ($maxWidth * $height / $maxHeight - $maxWidth) / 2;
                } else {
                    $xOffset = 0;
                    $yOffset = ($maxHeight * $width / $maxWidth - $maxHeight) / 2;
                }
                
                $newImg = imagecreatetruecolor($maxWidth,$maxHeight);
                
                imagecopyresampled($newImg,$img,0,0,$xOffset,$yOffset,$maxWidth,$maxHeight,$width-$xOffset,$height-$yOffset);
                
            } else {
                
                if ($width / $maxWidth > $height / $maxHeight) {
                    $newWidth = $maxWidth;
                    $newHeight = $height * $maxWidth / $width;
                } else {
                    $newHeight = $maxHeight;
                    $newWidth = $width * $maxHeight / $height;
                }
                
                $newImg = imagecreatetruecolor($newWidth,$newHeight);
                
                imagecopyresampled($newImg,$img,0,0,0,0,$newWidth,$newHeight,$width,$height);
                
            }
            
            ob_start();
            
            imagejpeg($newImg,'',100);
            
            $imageData = ob_get_contents();
            
            ob_end_clean();
            
        }
        
        return $imageData;

    }
    
    /**
    * creates the html to display the mugshot of a contact
    * 
    * @param string $class the css class of the img tag
    * @global Options used to determine, whether to show a nobody picture or nothing, if no picture is available
    * @return string html to display image
    */
    function html($class)
    {
        
        global $options;
        
        if ($this->contact->contact['pictureURL'])
            return '<img class="' . $class . '" src="' . $this->contact->contact['pictureURL'] . '" alt="' .
                $this->contact->contact['firstname'] . ' ' . $this->contact->contact['lastname'] . '" />';
                
        if ($this->contact->contact['pictureData'])
            return '<img class="' . $class . '" src="mugshot.php?id=' . $this->contact->contact['id'] . '" alt="' .
                $this->contact->contact['firstname'] . ' ' . $this->contact->contact['lastname'] . '" />';
        
        if ($options->getOption('picAlwaysDisplay'))
            return '<img class="' . $class . '" src="mugshots/nobody.jpg" alt="' .
                $this->contact->contact['firstname'] . ' ' . $this->contact->contact['lastname'] . '" />';
		
        return '';
        
    }
    
    /**
    * gets the type of the image, i.e. the way the image is saved
    * 
    * possible returns are 'file', 'data' or 'empty'
    * file and empty are straightforward, data means the image is saved in
    * the database as a blob
    * @return string type of image
    * @global Options used to determine, whether empty should be returned if no picture
    */
    function getType() {
        
        global $options;
        
        if ($this->contact->contact['pictureURL'])
            return 'file';
                
        if ($this->contact->contact['pictureData'])
            return 'data';
        
        if ($options->getOption('picAlwaysDisplay'))
            return 'file';
                
        return 'empty';
        
    }
    
    /**
    * returns the size of the image
    * 
    * @return array array of type 'width' => xy, 'height' => xy, or null if no image
    */
    function getSize() {
        
        if ($this->getType() == 'empty')
            return null;
            
        if ($this->getType() == 'file') {
            $tmp = @getimagesize($this->getData());
            if (!$tmp)
                return null;
            return array(
                'width' => $tmp[0],
                'height' => $tmp[1]
            );
        }
        
        $tmpFileName = tempnam('mugshots/','tmp');
        $tmpFile = fopen($tmpFileName,'wb');
        fwrite($tmpFile,$this->getData());
        fclose($tmpFile);
        $tmp = getimagesize($tmpFileName);
        unlink($tmpFileName);
        return array(
            'width' => $tmp[0],
            'height' => $tmp[1]
        );
        
    }
    
    /**
    * returns image data
    *
    * this function either returns the path of the image (file), a jpeg stream (data), or 
    * the empty string (empty), according to the value returned by {@link getType()}
    * @global Options used to determine, whether to show a nobody picture or nothing, if no picture is available
    * @global string used to transfer an absolute http url (i.e. /tab/mugshots) to a filesystem path (i.e. /srv/www/htdocs/tab/mugshots)
    * @return string filename or jpeg stream or empty string
    */
    function getData() {
        
        global $options, $CONFIG_ABSOLUTE_LOCAL_MUGSHOT_PATH;
        
        if ($this->contact->contact['pictureURL'])
            if (substr($this->contact->contact['pictureURL'],0,1) == '/')
                return $CONFIG_ABSOLUTE_LOCAL_MUGSHOT_PATH . substr(strrchr($this->contact->contact['pictureURL'],'/'),1);
            else
                return $this->contact->contact['pictureURL'];
                
        if ($this->contact->contact['pictureData'])
            return $this->contact->contact['pictureData'];
        
        if ($options->getOption('picAlwaysDisplay'))
            return 'mugshots/nobody.jpg';
                
        return '';
        
    }
}

?>
