<?php
/**
* contains class {@link PageContactEditNew}
* @author Tobias Schlatter, Thomas Katzlberger
* @package frontEnd
* @subpackage pages
*/

/** */
require_once('lib/backEnd/ErrorHandler.class.php');
require_once('lib/backEnd/Options.class.php');
require_once('lib/frontEnd/Page.class.php');
require_once('lib/backEnd/DB.class.php');
require_once('lib/backEnd/Contact.class.php');
require_once('lib/frontEnd/Navigation.class.php');
require_once('lib/utilities/RightsManager.class.php');

/**
* the contact edit page
* 
* the contact edit page allows users to
* add, edit and remove contact entries
* @package frontEnd
* @subpackage pages
*/
class PageContactEditNew extends Page {
    
    /**
    * @var Contact the contact that is to be modified
    */
    var $contact;
    
    /**
    * @var Navigation the menu to be shown
    */
    var $menu;
    
    /**
    * @var boolean whether the contact is to be added or to be changed
    */
    var $add;
    
    /**
    * @var int counter to create html ids for for scriptaculous
    */
    var $htmlId;
    
    var $counters; // private (counters array for value group POST)
    
    /**
    * Constructor
    * 
    * init superclass, init vars
    * @param integer|Contact the id of the contact, or the contact that is to be edited
    * @param boolean whether the contact is to be added or not (cannot be detected through {@link $id}, because a contact can be passed if an error occurs to preserve already inserted information)
    * @global Options admin options
    */
    function PageContactEditNew($id=null,$add=false) {

        global $options;
        
        $this->counters = array();
        
        $this->add = $add;
        
        if ($id === null)
            $this->contact = new Contact();
        elseif (is_numeric($id))
            $this->contact = new Contact($id);
        else
            $this->contact = &$id;
            
        if ($add)
            $this->Page('Add new entry');
        else
            $this->Page('Edit entry for ' . $this->contact->contact['firstname'] . ' ' . $this->contact->contact['lastname']);
            
        $this->menu = new Navigation();
        
        $this->menu->addEntry('save','save','javascript:saveEntry();');
        if (isset($this->contact->contact['id']))
            $this->menu->addEntry('cancel','cancel','contact.php?id=' . $this->contact->contact['id']);
        else
            $this->menu->addEntry('cancel','cancel','list.php');
            
        if (!$this->add) {
            if(RightsManager::mayDeleteContact($_SESSION['user'],$this->contact)) {
                $this->menu->addEntry('delete','delete','javascript:deleteEntry(' . $this->contact->contact['id'] . ');');
                if ($_SESSION['user']->isAtLeast('admin') && $options->getOption('deleteTrashMode'))
                    $this->menu->addEntry('trash','trash','contact.php?mode=trash&amp;id=' . $this->contact->contact['id']);
            }
        }
        
    }
    
    /**
    * create the content of admin panel
    * @return string html-content
    * @global ErrorHandler used for error handling
    * @global PluginManager used for plugin handling (i.e. additional editable fields)
    * @uses createNameFieldset()
    * @uses createAddressFieldset()
    * @uses createEmailFieldset()
    * @uses createPhoneFieldset()
    * @uses createMessagingFieldset()
    * @uses createWebsitesFieldset()
    * @uses createDatesFieldset()
    * @uses createOtherinfoFieldset()
    * @uses createMugshotFieldset()
    * @uses createNotesFieldset()
    * @uses createGroupsFieldset()
    */
    function innerCreate() {
        
        global $errorHandler,$pluginManager;
        
        $cont = '<div class="edit-container">';
                
        $cont .= $this->menu->create('edit-menu');
        
        if ($this->add)
            $cont .= '<div class="edit-title">Add new entry</div>';
        else
            $cont .= '<div class="edit-title">Edit entry for ' . $this->contact->contact['firstname'] . ' ' . $this->contact->contact['lastname'] . '</div>';
        
        $cont .= '<div class="edit-box">';
        $cont .= '<form enctype="multipart/form-data" name="editEntry" method="post" action="contact.php?mode=save">';
        
        if (!$this->add)
            $cont .= '<input type="hidden" name="contact[id]" value="' . $this->contact->contact['id'] . '" />';
        
        $cont .= $this->createNameFieldset();
        $cont .= $this->createAddressFieldset();
        
        $cont .= '<fieldset class="edit-additionals">';
        $cont .= '<legend>Communications</legend>';
        $cont .= $this->editHint();
        $opt = array('email'=>'email','phone'=>'phone','chat'=>'chat',' '=>'delete');
        $cont .= $this->createProperties($opt,'email');
        $cont .= $this->createProperties($opt,'phone');
        $cont .= $this->createProperties($opt,'chat');
        $cont .= $this->createProperties($opt,'blank');
        $cont .= '</fieldset>';

        $cont .= '<fieldset class="edit-additionals">';
        $cont .= '<legend>Information</legend>';
        $cont .= $this->editHint();
        $opt = array('other'=>'other','www'=>'www','date'=>'date',' '=>'delete');
        $cont .= $this->createProperties($opt,'www',12,37);
        $cont .= $this->createProperties($opt,'other',12,37);
        $cont .= $this->createProperties($opt,'date',12,37);
        $cont .= $this->createProperties($opt,'blank',12,37);
        $cont .= '</fieldset>';

        $cont .= '<fieldset class="edit-additionals">';
        $cont .= '<legend>Plugins</legend>';
        $cont .= '<div class="edit-line">';
        $cont .= $pluginManager->editContactInterface($this->contact, 'otherInfo'); // WARNING: This always calls methods in the OLD class PageContactEdit! 
        $cont .= '</div>';
        $cont .= '</fieldset>';
        $cont .= $pluginManager->editContactInterface($this->contact, 'ownFieldset'); // WARNING: This always calls methods in the OLD class PageContactEdit! 
        
        $cont .= $this->createMugshotFieldset();
        $cont .= $this->createNotesFieldset();
        $cont .= $this->createGroupsFieldset();
        
        $cont .= '</form>';
        $cont .= $this->menu->create('edit-menu');
        $cont .= '<br></div>';
        
        if(isset($this->contact->contact['lastUpdate']))
            $cont .= '<div class="update">This entry was last updated on ' . $this->contact->contact['lastUpdate'] . '</div>';
        
        $cont .= '</div>';
        
        return $cont;
        
    }
    
    /**
    * create the fieldset that allows user to edit other information
    * @global PluginManager used to get custom edit fields from plugins
    * @return string html-content
    */
    function createProperties($options,$type,$size1=6,$size2=43) {
        
        if(!array_key_exists($type,$this->counters))
            $this->counters[$type]=0;
                
        if($type=='blank')
        {
            $dummy = array('type'=>$type,'label'=>'','value'=>'','visibility'=>'visible');
            if($this->add==true)
                $vals = array($dummy,$dummy,$dummy,$dummy); // 4 blanks
            else
                $vals = array($dummy,$dummy); // 2 blanks
        }
        else
            $vals = $this->contact->getValueGroup($type);

        $cont = '';
        foreach ($vals as $m)
        {
            $c = $this->counters[$type];
            $cont .= "\n<div class='edit-propertyset'>" . $this->createDropdown2($type."[$c][type]",'',$options,$type);
            $cont .= "\n<input type='text' name='".$type."[$c][label]' value='{$m['label']}' size='$size1'/>\n";
            $cont .= "<input type='text' name='".$type."[$c][value]' value='{$m['value']}' size='$size2'/>\n";
            $cont .= $this->createDropdown2($type."[$c][visibility]",'',array('visible'=>'public','hidden'=>'private','admin-hidden'=>'admin'),$m['visibility']);
            $cont .= "<br></div>\n";
            $this->counters[$type]++;
        }
                
        return $cont;
        
    }

    
    /**
    * create the fieldset that allows user to select the groups of the contact, and whether the contact is hidden or not
    * @return string html-content
    * @global DB used to query database for groups
    */
    function createGroupsFieldset() {
        
        global $db;
        
        $cont = '<fieldset class="edit-groups">';
        $cont .= '<legend>Groups</legend>';
        
        $gr = $this->contact->getValueGroup('groups');
        
        $groups = array();
        
        foreach ($gr as $g)
            $groups[] = $g['groupname'];
        
        $db->query('SELECT * FROM ' . TABLE_GROUPLIST . ' ORDER BY groupname ASC');
        
        $cont .= '<div class="edit-line">';
        
        while ($r = $db->next())
            $cont .= $this->createCheckbox("groups[{$r['groupname']}]",$r['groupname'],in_array($r['groupname'],$groups));

        $cont .= '</div>';
        
        $cont .= '<div class="edit-line">';
        
        $cont .= $this->createTextField('newgroup','Add new group');
        $cont .= $this->createCheckbox('contact[hidden]','Hide this entry',(isset($this->contact->contact['hidden']) ? $this->contact->contact['hidden'] : 0),'edit-input-checkbox-hide');
        
        $cont .= '</div>';
        
        $cont .= '</fieldset>';
        
        return $cont;
        
    }
    
    /**
    * create the fieldset that allows user to edit the notes of the contact
    * @return string html-content
    */
    function createNotesFieldset() {
        
        $cont = '<fieldset class="edit-notes">';
        $cont .= '<legend>Notes</legend>';
        
        $cont .= '<div class="edit-hint" id="editHelp'.$this->htmlId.'"><a href="#" onclick="Element.hide(\'editHelp'.$this->htmlId.'\'); Effect.Appear(\'editHint'.$this->htmlId.'\',{duration:1.2}); return false;">?</a></div><div class="edit-hint" id="editHint'.($this->htmlId++).'" style="display: none;">Text in the Notes box will be displayed exactly as you type it. The area will grow after saving whenever you add more lines of text.</div>';
        
        $notes = isset($this->contact->contact['notes']) ? $this->contact->contact['notes'] : '';
        $n=count(explode("\n",$notes));
        $cont .= $this->createTextarea('contact[notes]',$notes,$n+2,70);
        
        $cont .= '</fieldset>';
        
        return $cont;
        
    }
    
    /**
    * create the fieldset that allows user to edit and eventually upload the mugshot of the contact
    * @global Options used to determine whether mugshot may be uploaded or not
    * @return string html-content
    */
    function createMugshotFieldset() {
       
        global $options;
        
        $cont = '<fieldset class="edit-mugshot">';
        $cont .= '<legend>Mugshot</legend>';

        $cont .= '<div class="edit-hint" id="editHelp'.$this->htmlId.'"><a href="#" onclick="Element.hide(\'editHelp'.$this->htmlId.'\'); Effect.Appear(\'editHint'.$this->htmlId.'\',{duration:1.2}); return false;">?</a></div><div class="edit-hint" id="editHint'.($this->htmlId++).'" style="display: none;">If an URL to the mugshot is set this overrides any uploaded picture! If you upload a picture to the DB make sure that the "URL to mugshot" field is empty and has no space in it. If the admin does not allow uploads the upload button will not show and you can only set an URL. In case of URLs you can omit the protocol and server name if the image is stored on the same server as this application runs. Then you could use /gallery/people/small/me.jpg to locate the image. PHP restrictions may prevent uploads.</div>';
        
        $cont .= $this->createTextField("contact[pictureURL]",'URL to mugshot',(isset($this->contact->contact['pictureURL']) ? $this->contact->contact['pictureURL'] : ''),'edit-mugshot-text');
        
        if ($options->getOption('picAllowUpload')) {
           $cont .= '<div class="edit-input-file">
                      <label for="contact[pictureData][file]">Upload mugshot</label>
                      <input type="file" name="contact[pictureData][file]" id="contact[pictureData][file]" />
                      </div>';
                      
           if (isset($this->contact->contact['pictureData']) && $this->contact->contact['pictureData'])
               $cont .= $this->createCheckbox('contact[pictureData][remove]','Remove current mugshot',false);
        }
        
        $cont .= '</fieldset>';
        
        return $cont;
    }    
   
    /**
    * create the fieldset that allows user to the names of the contact
    * @return string html-content
    */
    function createNameFieldset() {
        
        $cont = '<fieldset class="edit-names">';
        $cont .= '<legend>Names</legend>';
        
        $cont .= '<div class="edit-hint" id="editHelp'.$this->htmlId.'"><a href="#" onclick="Element.hide(\'editHelp'.$this->htmlId.'\'); Effect.Appear(\'editHint'.$this->htmlId.'\',{duration:1.2}); return false;">?</a></div><div class="edit-hint" id="editHint'.($this->htmlId++).'" style="display: none;">A Last Name/Company Name is required for an entry to exist. If you need an additional company name for a contact use the groups on the bottom of the page. A contact can be assigned to multiple groups.</div>';
        
        $cont .= $this->createTextField('contact[lastname]','Last name or company name', isset($this->contact->contact['lastname']) ? $this->contact->contact['lastname'] : '');
        $cont .= $this->createTextField('contact[firstname]','First name', isset($this->contact->contact['firstname']) ? $this->contact->contact['firstname'] : '');
        $cont .= $this->createTextField('contact[middlename]','Middle name', isset($this->contact->contact['middlename']) ? $this->contact->contact['middlename'] : '');
        $cont .= $this->createTextField("contact[nickname]",'Nickname', isset($this->contact->contact['nickname']) ? $this->contact->contact['nickname'] : '');
        
        $cont .= '</fieldset>';
        
        return $cont;
        
    }
    
    /**
    * create the fieldset that allows user to the addresses of the contact
    * @global array list of country names and acronyms
    * @global Options used to determine the country default
    * @return string html-content
    */
    function createAddressFieldset() {
        
        global $country, $options;
        
        $cont = '<fieldset class="edit-address">';
        $cont .= '<legend>Addresses</legend>';
        
        $cont .= '<div class="edit-hint" id="editHelp'.$this->htmlId.'"><a href="#" onclick="Element.hide(\'editHelp'.$this->htmlId.'\'); Effect.Appear(\'editHint'.$this->htmlId.'\',{duration:1.2}); return false;">?</a></div><div class="edit-hint" id="editHint'.($this->htmlId++).'" style="display: none;">All fields are optional. An address will be saved if either of the following are provided: type, address lines, city, state, zip code, or a phone number. If Primary Address is selected, the address will be displayed in the contact list. To obtain more than 2 additional blank address sections, save this entry and edit it again.</div>';
        
        $addr = $this->contact->getValueGroup('addresses');
        
        $n = max(count($addr),1); // initially 3 blank adresses
        
        // write out all existing addresses plus 2 blank address
        for($i=0; $i<$n+2; $i++) {
            
            $a = (array_key_exists($i,$addr) ? $addr[$i] : null); // generate additional blank entries 
            
            if($i>=$n) // hide the blank addresses by default
                $cont .= '<div class="edit-single-address" id="anotherAddress'.$i.'" style="display: none;"><br>';
            else
                $cont .= '<div class="edit-single-address">';
            
            if (!$this->add)
                $cont .= '<input type="hidden" name="address[' . $i . '][refid]" value="' . $a['refid'] . '" />';
            
            $cont .= '<div class="edit-line">';
            $cont .= $this->createTextField("address[$i][type]",'Type',$a['type']);
            $cont .= $this->createRadioButton("address_primary",'Set as primary address',$i,isset($this->contact->contact['primaryAddress']) && $a['refid'] == $this->contact->contact['primaryAddress']);
            if(!$i>=$n)
                $cont .= '<div class="edit-input-link"><a href="javascript:deleteAddress(' . $i . ');">Delete this address</a></div>';
            $cont .= '</div>';
            
            $cont .= '<div class="edit-line">';
            $cont .= $this->createTextField("address[$i][line1]",'Address (Line 1)',$a['line1']);
            $cont .= $this->createTextField("address[$i][line2]",'Address (Line 2)',$a['line2']);
            $cont .= '</div>';
            
            $cont .= '<div class="edit-line">';
            $cont .= $this->createTextField("address[$i][city]",'City',$a['city']);
            $cont .= $this->createTextField("address[$i][state]",'State',$a['state']);
            $cont .= $this->createTextField("address[$i][zip]",'Zip-code',$a['zip']);
            $cont .= '</div>';
            
            $cont .= '<div class="edit-line">';
            $cont .= $this->createTextField("address[$i][phone1]",'Phone 1',$a['phone1']);
            $cont .= $this->createTextField("address[$i][phone2]",'Phone 2',$a['phone2']);
            $cont .= $this->createDropdown("address[$i][country]",'Country',$country,$a['country']);
            $cont .= '</div>';
            
            if($i>=$n-1 && $i!=$n+1) // include the add more link in the previous div
                $cont .= "\n".'<div id="addAddressLink'.($i+1).'"><a href="#" onclick="Element.hide(\'addAddressLink'.($i+1).'\'); Effect.SlideDown(\'anotherAddress'.($i+1).'\',{duration:1.2}); return false;">add address</a></div>';
                            
            $cont .= '</div>';            
        }
                
        $cont .= '</fieldset>';
        
        return $cont;
        
    }
    
    /**
    * create a textarea for an arbitrary value group
    *
    * the values of the group are separated by |'s
    * @param string $type which value group of the contact is to be shown
    * @param integer $cols how many columns should the textarea have
    * @return string html-content
    */
    function createValueGroupTextarea($type,$cols) {
        
        $v = '';
        $vals = $this->contact->getValueGroup($type);
        
        foreach ($vals as $m)
            $v .= "{$m['value']}|{$m['label']}" . ($m['visibility']=='hidden'?'|h':'') . ($m['visibility']=='admin-hidden'?'|a':'') ."\n";
        
        $cont .= $this->createTextarea($type,$v,count($vals)+2,$cols);
        
        return $cont;
        
    }
    
    /**
    * create a textfield
    *
    * @param string $name which name is to be used for the textfield
    * @param string $caption what caption should the textfield have
    * @param string $val default value of the textfield
    * @return string html-content
    * @static
    */
    function createTextField($name,$caption,$val=null,$class="edit-input") {
        
        $cont = "<div class='$class'>";
        $cont .= '<label for="' . $name . '">' . $caption . '</label>';
        $cont .= '<input type="text" name="' . $name . '" id="' . $name . '" ' . ($val!==null?'value="' . $val . '" ':'') . 'size=60 />'; // extends to available space set by CSS style up to 80
        $cont .= '</div>';
        
        return $cont;
        
    }
    
    /**
    * create a single radio button
    *
    * @param string $name which name is to be used for the button (respectively it's group)
    * @param string $caption what caption should the button have
    * @param string $val value of the button's group, if selected
    * @param boolean $sel is radio button selected
    * @return string html-content
    * @static
    */
    function createRadioButton($name,$caption,$val,$sel=false) {
     
        static $idNrs = array();
        
        if (!isset($idNrs[$name]))
            $idNrs[$name] = 0;
        else
            $idNrs[$name]++;
        
        $cont = '<div class="edit-input-radio">';
        $cont .= '<input type="radio" name="' . $name .'" id="' . $name . $idNrs[$name] . '" value="' . $val . '" ' . ($sel?'checked="checked" ':'') . '/>';
        $cont .= '<label for="' . $name . $idNrs[$name] . '">' . $caption . '</label>';
        $cont .= '</div>';
        
        return $cont;
        
    }
    
    /**
    * create a dropdown list
    *
    * @param string $name which name is to be used for the dropdown
    * @param string $caption what caption should the dropdown have
    * @param array $sels (associative) array with of type: value => caption
    * @param string $cur value of the item that should be pre-selected
    * @param boolean $nval whether the captions of {@link $sels} should be used as values or not
    * @return string html-content
    * @static
    */
    function createDropdown($name,$caption,$sels,$cur,$nval=false) {
        
        $cont = "<div class='edit-input'>";
        $cont .= '<label for="' . $name . '">' . $caption . '</label>';
        $cont .= '<select name="' . $name . '" id="' . $name . '">';
        
        foreach($sels as $k => $s) {
            if ($nval)
                $k = $s;
            $cont .= '<option value="' . $k . '"' . ($k === $cur?' selected="selected"':'') . '>' . $s . '</option>';
        }
        
        $cont .= '</select>';
        $cont .= '</div>';
        
        return $cont;
        
    }
    
    /**
    * create a dropdown list
    *
    * @param string $name which name is to be used for the dropdown
    * @param string $caption what caption should the dropdown have
    * @param array $sels (associative) array with of type: value => caption
    * @param string $cur value of the item that should be pre-selected
    * @param boolean $nval whether the captions of {@link $sels} should be used as values or not
    * @return string html-content
    * @static
    */
    function createDropdown2($name,$caption,$sels,$cur,$nval=false) {
        
        //$cont .= '<label for="' . $name . '">' . $caption . '</label>';
        $cont = '<select name="' . $name . '" id="' . $name . '">';
        
        foreach($sels as $k => $s) {
            if ($nval)
                $k = $s;
            $cont .= '<option value="' . $k . '"' . ($k === $cur?' selected="selected"':'') . '>' . $s . '</option>';
        }
        
        $cont .= '</select>';
        
        return $cont;
        
    }
    
    /**
    * create a textarea
    *
    * @param string $name which name is to be used for the textarea
    * @param string $cnt content of the textarea
    * @param integer $cols how many rows should the textarea have
    * @param integer $cols how many columns should the textarea have
    * @return string html-content
    */
    function createTextarea($name,$cnt,$rows,$cols) {
        
        $cont = '<textarea cols="' . $cols . '" rows="' . $rows . '" name="'. $name . '">';
        $cont .= $cnt;
        $cont .= '</textarea>';
        
        return $cont;
        
    }
    
    /**
    * create a single checkbox
    *
    * @param string $name which name is to be used for the checkbox
    * @param string $caption what caption should the checkbox have
    * @param boolean $sel is checkbox selected
    * @param string $cssclass CSS class; default: edit-input-checkbox
    * @return string html-content
    * @static
    */
    function createCheckbox($name,$caption,$sel=false,$cssclass='edit-input-checkbox') {
        
        $cont = "<div class='$cssclass'>";
        $cont .= '<input type="hidden" name="' . $name .'" id="' . $name . '" value="0" />';
        $cont .= '<input type="checkbox" name="' . $name .'" id="' . $name . '" value="1" ' . ($sel?'checked="checked" ':'') . '/>';
        $cont .= '<label for="' . $name . '">' . $caption . '</label>';
        $cont .= '</div>';
        
        return $cont;
        
    }
    
    // private
    function editHint()
    {
        return '<div class="edit-hint" id="editHelp'.$this->htmlId.'"><a href="#" onclick="Element.hide(\'editHelp'.$this->htmlId.'\'); Effect.Appear(\'editHint'.$this->htmlId.'\',{duration:1.2}); return false;">?</a></div><div class="edit-hint" id="editHint'.($this->htmlId++).'" style="display: none;">
        <div>Each entry is of the form: group, label, value, and view permissions. The group selects where information is placed on the output. For example all emails are placed together. Select the label to qualify an entry. For example for phonenumbers you could use fax or cell. Viewing permissions can be 
        public (everyone can see the item), private (only the user can see and modify the item), admin (user can view but only an admin can modify an item).</div></div>';        
    }
}



?>
