<?php
/**
* contains class {@link PageList}
* @author Tobias Schlatter
* @package frontEnd
* @subpackage pages
*/

/** */
require_once('lib/backEnd/DB.class.php');
require_once('lib/frontEnd/Page.class.php');
require_once('lib/frontEnd/pages/PageContact.class.php');
require_once('lib/frontEnd/pages/PageRecentlyChanged.class.php');
require_once('lib/frontEnd/pages/PageUpcomingDates.class.php');
require_once('lib/frontEnd/Navigation.class.php');
require_once('lib/backEnd/GroupContactList.class.php');
require_once('lib/utilities/TableGenerator.class.php');
require_once('lib/utilities/GroupNormalizer.class.php');

/**
* the list page
* 
* the list page is the main page of TAB.
* it shows links to other features of TAB and displays a list
* of all the contacts.
* @package frontEnd
* @subpackage pages
*/
class PageList extends Page {
    
    /**
    * @var GroupContactList list of contacts to display
    */
    var $contactList;
    
    /**
    * @var Navigation main navigation of list page
    */
    var $menu;
    
    /**
    * @var boolean whether to expand contact entries or not
    */
    var $expand;
    
    /**
    * Constructor
    *
    * init {@link $contactList}, and menu
    * @param string $group group to display list for
    * @param boolean $expand whether to contact entries or not
    * @param string $begin the point in the alphabet, where the list should start (does not have to be just one character 'Sch' is also ok)
    * @param integer $page number of page to show
    * @global Options used to determine how many entries per page should be shown
    * @global PluginManager used for menu completion
    * @global string name of popup window (unused)
    * @todo remove or implement popup windows
    */
    function PageList($group='',$expand=false,$begin='',$page=0) {
        global $options,$pluginManager;
        
        $this->Page('List entries');
        
        $this->contactList = new GroupContactList($group);
        $this->contactList->setStartString($begin);
        $this->contactList->setPage($page);
        $this->contactList->setEntriesPerPage($options->getOption('limitEntries'));
        
        $this->expand = $expand;
        
        $this->menu = new Navigation();
                
        $this->menu->addEntry('search','search','searchlist.php');
        
        if ($_SESSION['user']->isAtLeast('user')) {
            $this->menu->addEntry('add','add new entry','contact.php?mode=new');
            $this->menu->addEntry('settings','personal settings','options.php');
            //$this->menu->addEntry('mail','mailing list','mailto.php?group=' . $group);
            //$this->menu->addEntry('export','export','export.php');
            if ($_SESSION['user']->isAtLeast('admin'))
                $this->menu->addEntry('admin','admin panel','adminPanel.php');
        }
                
        $pluginManager->listMenu($this->contactList,$this->menu);
        
    }
    
    /**
    * create the content of admin panel
    * @return string html-content
    * @global boolean whether to display whole address in the list (unused)
    * @todo implement address display, or remove completely
    * @global Options used to determine several options
    * @uses PageUpcomingDates
    * @uses PageRecentlyChanged
    * @uses createGOTO()
    * @uses createGroupSelector()
    * @uses createPageNav()
    * @uses createCharNav()
    * @uses createTable()
    */
    function innerCreate() {
    
        global $options,$CONFIG_LIST_BANNER,$CONFIG_LIST_RECENTLY_CHANGED_LIMIT;
    
        $cont = '<div class="contact-list"><a name="top" id="top">'.$CONFIG_LIST_BANNER.'</a>';
        
        $cont .= $this->createInfoBox();
        
        $cont .= '<div class="list-box">';
        if ($options->getOption('bdayDisplay')) {
            // display dates and changed list
                $cont .= '<div class="upcoming-dates">';
                    $dates = new PageUpcomingDates();
                    $cont .= $dates->innerCreate();
                $cont .= '</div>';
        }
        if ($options->getOption('recentlyChangedDisplay')) {
                $cont .= '<div class="recently-changed">';
                    $changed = new PageRecentlyChanged($options->getOption('recentlyChangedLimit')); // display at most 8
                    $cont .= $changed->innerCreate(true);
                $cont .= '</div>';
        }
        $cont .= '</div>';
        
        $cont .= $this->menu->create('list-menu');
        
        $cont .= $this->createGOTO();
        $cont .= $this->createGroupSelector();
        
        $cont .= '<div class="list-nav">';
        $cont .= $this->createPageNav();
        $cont .= $this->createCharNav();
        $cont .= '</div>';
        
        $cont .= $this->createTable();
        
        $cont .= '</div>';
        
        return $cont;
        
    }
    
    /**
    * create the goto text-box
    * @return string html-content
    */
    function createGOTO() {
        global $CONFIG_SEARCH_CUSTOM;
        
        $cont = '<div class="goTo">';
        
        $cont .= '<form name="goToEntry" method="post" action="search.php">';
        
        $cont .= '<label for="goTo">go to</label>';
        
        $cont .= '<input type="text" name="goTo" id="goTo" />';
        
        $cont .= '<select name="type" onchange="var v=this.options[this.selectedIndex].value; createCookie(\'searchtype\',v,365);">';
        
        $cont .= '<option value="name">Name</option>';
        $cont .= '<option value="email">E-Mail addresses</option>';
        $cont .= '<option value="www">Websites</option>';
        $cont .= '<option value="chat">Messaging</option>';
        $cont .= '<option value="address">Addresses</option>';
        $cont .= '<option value="phone">Phonenumbers</option>';
        
        $n=count($CONFIG_SEARCH_CUSTOM);
        for($i=0;$i<$n;$i++)
            $cont .= "<option value='custom_$i'>".$CONFIG_SEARCH_CUSTOM[$i].'</option>';
        
        $cont .= '</select>';
        
        $cont .= '</form>';
        
        $cont .= '<div id="autocompletegoto-hint"></div>';
        
        $cont .= <<<EOC
<script type="text/javascript">
<!--
new Ajax.Autocompleter("goTo","autocompletegoto-hint","autocompletegoto.ajax.php");
document.goToEntry.goTo.focus();
createCookie("searchtype","name",365);
//-->
</script>
EOC;
        
        $cont .= '</div>';
        
        return $cont;
        
    }
    
    /**
    * create the info-box
    *
    * the info box shows the user which is currently logged in,
    * it also shows a short description of the user rights and
    * the custom welcome-message
    * @return string html-content
    * @global Options used to determine the welcome message
    */
    function createInfoBox() {
        
        global $options;
        
        // create info box
        $cont = '<div class="info-box">';
        
        if ($options->getOption('msgWelcome') != '')
            $cont .= '<div class="welcome-message">' . $options->getOption('msgWelcome') . '</div>';
        
        if (is_a($_SESSION['user'],'GuestUser'))
            $cont .= '<div class="login-status">You are not logged in. <a href="index.php?mode=login">Log in here</a></div>';
        else {
            $cont .= '<div class="login-status">You are currently logged in as: ' . $_SESSION['user']->contact['firstname'] . ' ' . $_SESSION['user']->contact['lastname'] . '</div>';
            if ($_SESSION['user']->isAtLeast('administrator'))
                $cont .= '<div class="access-status">You have administrator access.</div>';
            elseif ($_SESSION['user']->isAtLeast('manager'))
                $cont .= '<div class="access-status">You have manager access.</div>';
            elseif ($_SESSION['user']->isAtLeast('user'))
                $cont .= '<div class="access-status">You have normal user access.<br>Entries you may modify are displayed in green.</div>';
            $cont .= '<div class="logout"><a href="index.php?mode=logout">Log out here.</a></div>';
        }
        $cont .= '</div>';
        
        return $cont;
        
    }
    
    /**
    * create the page navigation
    *
    * the page navigation is only shown, if the currently selected character
    * contains more entries, than fit on one page
    * @return string html-content
    */
    function createPageNav() {
        
        $count = $this->contactList->getBoundedPageCount();
        
        $cont = '';
        
        if ($count > 1) {
            
            $showCount = 2;
            $distThreshold = 10;
            $cur = $this->contactList->page;
            
            for ($i=$cur-$showCount;$i<=$cur+$showCount;$i++)
                if ($i >= 0 && $i < $count)
                    $cont .= '<a href="list.php?group=' . $this->contactList->group . '&amp;begin=' . rawurlencode($this->contactList->start) . '&amp;page=' . $i . '"' . ($i==$this->contactList->page?' class="curpage"':'') . '>[' . intval($i+1) . ']</a>';
            
            $i = floor($cur/2);
            if ($cur >= $distThreshold)
                $cont = '<a href="list.php?group=' . $this->contactList->group . '&amp;begin=' . rawurlencode($this->contactList->start) . '&amp;page=' . $i . '"' . ($i==$this->contactList->page?' class="curpage"':'') . '>[' . intval($i+1) . ']</a>' .
                        '...' . $cont;
                
            $i = 0;
            if ($cur-$showCount > 1)
                $cont = '<a href="list.php?group=' . $this->contactList->group . '&amp;begin=' . rawurlencode($this->contactList->start) . '&amp;page=' . $i . '"' . ($i==$this->contactList->page?' class="curpage"':'') . '>[' . intval($i+1) . ']</a>' .
                        '...' . $cont;
                        
            if ($cur-$showCount == 1)
                $cont = '<a href="list.php?group=' . $this->contactList->group . '&amp;begin=' . rawurlencode($this->contactList->start) . '&amp;page=' . $i . '"' . ($i==$this->contactList->page?' class="curpage"':'') . '>[' . intval($i+1) . ']</a>' .
                        $cont;
            
            $i = ceil(($count-1+$cur)/2);
            if ($count-$cur > $distThreshold)
                $cont .= '...<a href="list.php?group=' . $this->contactList->group . '&amp;begin=' . rawurlencode($this->contactList->start) . '&amp;page=' . $i . '"' . ($i==$this->contactList->page?' class="curpage"':'') . '>[' . intval($i+1) . ']</a>';
            
                        
            $i = $count-1;
            if ($cur+$showCount < $i-1)
                $cont .= '...<a href="list.php?group=' . $this->contactList->group . '&amp;begin=' . rawurlencode($this->contactList->start) . '&amp;page=' . $i . '"' . ($i==$this->contactList->page?' class="curpage"':'') . '>[' . intval($i+1) . ']</a>';
            
            if ($cur+$showCount == $i-1)
                $cont .= '<a href="list.php?group=' . $this->contactList->group . '&amp;begin=' . rawurlencode($this->contactList->start) . '&amp;page=' . $i . '"' . ($i==$this->contactList->page?' class="curpage"':'') . '>[' . intval($i+1) . ']</a>';
            
            
            $cont = '<div class="page-nav">' . $cont;
            $cont .= '</div>';
        
        }
        
        return $cont;
        
    }
    
    /**
    * create the char navigation
    *
    * the char navigation shows a link for each first character which is
    * in the contact list. this allows easy navigation even in big lists
    * @return string html-content
    * @global GroupNormalizer needed to convert characters in their basic form (i.e. to convert ö to o)
    */
    function createCharNav() {
        
        global $groupNormalizer;
        
        $chars = $this->contactList->getFirstChars();
        $start = $groupNormalizer->normalize(mb_strtolower(mb_substr($this->contactList->start,0,1)));
        $finish = $groupNormalizer->normalize(mb_strtolower(mb_substr($this->contactList->getRightStringBound(),0,1)));
        $rf = ($finish != mb_strtolower(mb_substr($this->contactList->getRightOuterStringBound(),0,1)));
        $in = false;

        $cont = '<ul class="charNav">';
        
        foreach ($chars as $char) {
            $class = '';
            if ($start != '') {
                if ($groupNormalizer->normalize(mb_strtolower($char)) == $start) {
                    $class = 'cur';
                    $in = true;
                }
                if ($groupNormalizer->normalize(mb_strtolower($char)) == $finish) {
                    $in = false;
                    if ($finish == $start) {
                        // don't change class from start
                    } elseif ($rf) {
                        $class = 'obs';
                    } else
                        $class = 'tail';
                }
                if ($in && $groupNormalizer->normalize(mb_strtolower($char)) != $start)
                    $class = 'obs';
            }
                
            $cont .= '<li' . ($class?' class="' . $class . '"':'') . '><a href="list.php?group=' . $this->contactList->group . '&amp;begin=' . rawurlencode($char) . '">' . $groupNormalizer->normalize(mb_strtoupper($char)) . '</a></li>';
                
        }
        
        $class = '';
        
        if (!$this->contactList->start)
            $class = 'cur';
        
        $cont .= '<li' . ($class?' class="' . $class . '"':'') . '><a href="list.php?group=' . $this->contactList->group . '&amp;begin=">[all]</a></li>';
    
        if ($this->expand)
            $cont .= '<li><a href="list.php?group=' . $this->contactList->group .
                '&amp;begin=' . $this->contactList->start .
                '&amp;page=' . $this->contactList->page . '">collapse</a></li>';
        else
            $cont .= '<li><a href="list.php?group=' . $this->contactList->group .
                '&amp;begin=' . $this->contactList->start .
                '&amp;page=' . $this->contactList->page .
                '&amp;expand=1">expand</a></li>';
            
        $cont .= '</ul>';
        
        return $cont;
        
    }
    
    /**
    * create the dropdown for group selection
    *
    * @return string html-content
    * @global DB used to query database for groups
    */
    function createGroupSelector() {
        
        global $db;
        
        // create group selector
        $cont = '<div class="group-selector">';
        $cont .= '<form name="selectGroup" method="get" action="list.php">';
        $cont .= '<label for="group">select group</label>';
        $cont .= '<select name="group" id="group" onchange="document.selectGroup.submit()">';
        
        $cont .= '<option value=""' . ($this->contactList->group == ''?' selected="selected"':'') . '>(all entries)</option>';
        
        if ($_SESSION['user']->isAtLeast('admin'))
            $cont .= '<option value="hidden"' . ($this->contactList->group == 'hidden'?' selected="selected"':'') . '>(hidden entries)</option>';
            
        $cont .= '<option value="ungrouped"' . ($this->contactList->group == 'ungrouped'?' selected="selected"':'') . '>(ungrouped entries)</option>';
        
        $db->query('SELECT * FROM ' . TABLE_GROUPLIST . ' WHERE groupname NOT LIKE "#h#%" ORDER BY groupname ASC');
        
        while ($r = $db->next())
            $cont .= '<option' . ($this->contactList->group == $r['groupname']?' selected="selected"':'') . '>' . $r['groupname'] . '</option>';
        
        $cont .= '</select>';
        $cont .= '</form>';
        $cont .= '</div>';
        
        return $cont;
        
    }
    
    /**
    * create the table containing the contacts
    *
    * @uses Contact
    * @return string html-content
    * @global GroupNormalizer used to modify the contact names, in order to get them correctly grouped
    * @uses TableGenerator
    */
    function createTable() {
        
        global $groupNormalizer;
        
        // create big table
        $contacts = $this->contactList->getContacts();
        
        $data = array();
        
        foreach($contacts as $c) {
            
            if ($this->expand) {
                
                $p = new PageContact($c);
                
                $data[] = array(
                    'cont' => $p->innerCreate(),
                    'class' => 'list-expanded-card',
                    'group' => $groupNormalizer->normalize(mb_substr($c->contact['lastname'],0,1))
                );
                
            } else {
                $add = $c->getValueGroup('addresses');
                
                $mails = $c->getValueGroup('email');
                
                $mail = '';
                foreach ($mails as $m) {
                    $mail .= $c->generateEmail($m) . '<br />';
                }
                
                if ($_SESSION['user']->contact['id'] == $c->contact['id'])
                    $class = 'me';
                else if ($_SESSION['user']->getType() == 'user' && $c->contact['whoAdded'] == $_SESSION['user']->id)
                    $class = 'editable';
                else
                    $class = '';
                
                $data[] = array(
                    'display_name' => '<a href="contact.php?id=' . $c->contact['id'] . '">' . $c->contact['lastname'] . (!empty($c->contact['firstname']) ? ', ' . $c->contact['firstname'] : '') . '</a>',
                    'mails' => $mail,
                    'phone' => (array_key_exists(0,$add) ? $add[0]['phone1'] : ''),
                    'class' => $class,
                    'group' => $groupNormalizer->normalize(mb_substr($c->contact['lastname'],0,1))
                );
            }
            
        }
        
        $tGen = new TableGenerator('contact-list');
        
        $cont = '<table class="contact-list">';
        
        $cont .= '<caption>' . $this->contactList->getGroupCaption() . '</caption>';
        
        if (count($data) > 0)
            if ($this->expand)
                $cont .= $tGen->generateBody($data,array('cont'),'class','group');
            else 
                $cont .= $tGen->generateBody($data,array('display_name','phone','mails'),'class','group');
        else
            $cont .= '<tr class="noentry"><td>No Entries.</td></tr>';
        
        $cont .= '</table>';
        
        return $cont;
        
    }
    
}

?>
