<?php
/**
* contains class {@link PageSearchList}
* @author Thomas Katzlberger
* @package frontEnd
* @subpackage pages
*/

/** */
require_once('lib/backEnd/DB.class.php');
require_once('lib/frontEnd/Page.class.php');
require_once('lib/frontEnd/pages/PageContact.class.php');
require_once('lib/utilities/TableGenerator.class.php');

/**
* the search list page
* 
* the search list page allows searches similar to the autocomplete function.
* The page usually displays the full search result, although it can limit the result (without navigation)
* @package frontEnd
* @subpackage pages
*/
class PageSearchList extends Page {
    
    /**
    * @var ContactList list of contacts to display
    */
    var $contactList;
        
    /**
    * @var boolean whether to expand contact entries or not
    */
    var $expand;

    /**
    * @var last searchstring
    */
    var $search;

    /**
    * @var last searchtype
    */
    var $searchtype;

    /**
    * @var Navigation basic nav menu: return (to list) and expand
    */
    var $nav;

    /**
    * Constructor
    *
    * init {@link $contactList}, and menu
    * @param string $search part of a searchtype (e.g. name or email)
    * @param string $searchtype [name|email|www|chat|phone|custom_?] Custom searches defined in config.php, shared with autocomplete.
    * @param boolean $exp whether to expand entries or not
    * @param integer $number limit of entries (default = 0 which means unlimited) 
    */
    function PageSearchList($search,$searchtype,$exp=false,$number=0)
    {            
        $this->Page('Search List');
        
        $this->search = $search;
        $this->searchtype = $searchtype;
        $this->expand = $exp;

        $this->nav = new Navigation();
        $this->nav->addEntry('expand','expand','searchlist.php?search=' . $search .
                '&amp;type=' . $searchtype . '&amp;expand=1');
        $this->nav->addEntry('return','return','list.php');
        
        $sql=$this->createQuery($search,$searchtype);
        $this->contactList = new ContactList($sql);
        $this->contactList->setEntriesPerPage($number);
    }

    /**
    * Create search query
    *
    * init {@link $contactList}, and menu
    * @param search $search partial string to match
    * @param searchtype $searchtype [name|email|www|chat|phone|custom_?] Custom searches defined in config.php, shared with autocomplete.
    * @global array custom searchtypes defined in config.php
    * @global DB used for database access
    */
    function createQuery($search,$searchtype) {
        global $CONFIG_SEARCH_CUSTOM, $db;
        $sql="SELECT * FROM " . TABLE_CONTACT . " AS contact WHERE id=-1"; // create an empty result - any better way to do this??
                                                                       
        $admin = intval($_SESSION['user']->isAtLeast('admin'));
        
        if($search=='')
            return $sql;
       
        $p = $search;
              
        switch ($searchtype)
        {
            case 'name':
                $sel1 = "SELECT * FROM " . TABLE_CONTACT . " AS contact WHERE ";
                $sel2 = "SELECT * FROM " . TABLE_CONTACT . " AS contact WHERE ";
                $where1 = "(lastname LIKE '$p%') AND (hidden = 0 OR $admin) ORDER BY lastname";
                $where2 = "(firstname LIKE '$p%' OR nickname LIKE '$p%') AND (hidden = 0 OR $admin) ORDER BY firstname";
                $sql = "($sel1 $where1) UNION ($sel2 $where2)";
                break;
            case 'email':
            case 'www':
            case 'chat':
                $sel = "SELECT *
                FROM " . TABLE_CONTACT . " AS contact, " . TABLE_PROPERTIES . " AS properties WHERE ";
                $where = "contact.id=properties.id AND properties.type = " . $db->escape($type) . " AND properties.value LIKE '%$p%' AND (properties.visibility = 'visible' OR $admin) AND (contact.hidden = 0 OR $admin) ORDER BY lastname";
                $sql = "$sel $where";
                break;
            case 'phone':
                $sel1 = "SELECT contact.* FROM " . TABLE_CONTACT . " AS contact, " . TABLE_PROPERTIES . " AS properties WHERE ";
                $where1 = "contact.id=properties.id AND properties.type = " . $db->escape($type) . " AND properties.value LIKE '%$p%' AND (properties.visibility = 'visible' OR $admin) AND (contact.hidden = 0 OR $admin) ORDER BY lastname";
                $sel2 = "SELECT contact.* FROM " . TABLE_CONTACT . " AS contact, " . TABLE_ADDRESS . " AS address WHERE ";
                $where2 = "contact.id=address.id AND (phone1 like '%$p%' OR phone2 like '%$p%') AND (hidden = 0 OR $admin) ORDER BY lastname";
                $sql = "($sel1 $where1) UNION ($sel2 $where2)";
                break;
            case 'address':
                $sel = "SELECT * FROM " . TABLE_CONTACT . " AS contact, " . TABLE_ADDRESS . " AS address WHERE ";
                $where = "contact.id=address.id AND (line1 like '%$p%' OR line2 like '%$p%' OR city like '%$p%') AND (hidden = 0 OR $admin) ORDER BY lastname";
                $sql = "$sel $where";
                break;
            default:
                $n=count($CONFIG_SEARCH_CUSTOM);
                for($i=0;$i<$n;$i++)
                {
                    if($searchtype=="custom_$i")
                    {
                        $sel = "SELECT *
                        FROM " . TABLE_CONTACT . " AS contact, " . TABLE_PROPERTIES . " AS properties WHERE ";
                        $where = "contact.id=properties.id AND properties.type = 'other' AND properties.label = '".$CONFIG_SEARCH_CUSTOM[$i]."' AND properties.value LIKE '%$p%' AND (properties.visibility = 'visible' OR $admin) AND (contact.hidden = 0 OR $admin) ORDER BY lastname";
                        $sql = "$sel $where";
                        break;
                    }
                }
                break;
        }
        
        return $sql;
    }
    
    /**
    * create the content of the search list
    * @return string html-content
    */
    function innerCreate() {
        
        $cont = '<div class="contact-list">';
                
        $cont .= $this->createSearchInterface();
        $cont .= $this->nav->create('options-menu');        
        $cont .= $this->createTable();
        
        $cont .= '</div>';
        
        return $cont;        
    }
    
    /**
    * create the goto text-box
    * @return string html-content
    * @global array custom searchtypes defined in config.php
    */
    function createSearchInterface() {
        global $CONFIG_SEARCH_CUSTOM;
        
        $cont = "\n<div class='goTo'>";
        
        $cont .= '<form name="search" method="get" action="searchlist.php">';
        
        $cont .= '<label for="goTo">Search</label>';
        
        $cont .= '<input type="text" name="search" id="search" value="'.$this->search.'"/>';
        
        $cont .= '<select name="type">';
        
        $cont .= '<option value="name"    '.($this->searchtype=='name' ? 'selected="selected"' : '').'>Name</option>';
        $cont .= '<option value="email"   '.($this->searchtype=='email' ? 'selected="selected"' : '').'>E-Mail addresses</option>';
        $cont .= '<option value="www"     '.($this->searchtype=='www' ? 'selected="selected"' : '').'>Websites</option>';
        $cont .= '<option value="chat"    '.($this->searchtype=='chat' ? 'selected="selected"' : '').'>Messaging</option>';
        $cont .= '<option value="address" '.($this->searchtype=='address' ? 'selected="selected"' : '').'>Addresses</option>';
        $cont .= '<option value="phone"   '.($this->searchtype=='phone' ? 'selected="selected"' : '').'>Phonenumbers</option>';
        
        $n=count($CONFIG_SEARCH_CUSTOM);                                               
        for($i=0;$i<$n;$i++)
            $cont .= "<option value='custom_$i' ".($this->searchtype=="custom_$i" ? 'selected="selected"' : '').">".$CONFIG_SEARCH_CUSTOM[$i].'</option>';
        
        $cont .= '</select>';
        $cont .= '<input type="submit" value="Go" id="submit" />';
        $cont .= '</form>';                
        $cont .= '</div><div><br><br></div>'; // make some space ...
        
        return $cont;
    }
        
    /**
    * create the table containing the contacts
    *
    * @uses Contact
    * @return string html-content
    * @uses TableGenerator
    * @todo Use the search results to send emails, output depending on query
    */
    function createTable() {
                
        // create big table
        $contacts = $this->contactList->getContacts();
        
        $data = array();
        
        foreach($contacts as $c) {
            
            if ($this->expand) {
                
                $p = new PageContact($c);
                
                $data[] = array(
                    'cont' => $p->innerCreate(),
                    'class' => 'list-expanded-card');
                
            } else {
                $add = $c->getValueGroup('addresses');
                
                $mails = $c->getValueGroup('email');
                
                $mail = '';
                foreach ($mails as $m) {
                    $mail .= $c->generateEmail($m) . '<br />';
                }
                
                if ($_SESSION['user']->contact['id'] == $c->contact['id'])
                    $class = 'me';
                else if ($_SESSION['user']->getType() == 'user' && $c->contact['whoAdded'] == $_SESSION['user']->id)
                    $class = 'editable';
                else
                    $class = '';
                
                $data[] = array(
                    'display_name' => '<a href="contact.php?id=' . $c->contact['id'] . '">' . $c->contact['lastname'] . ', ' . $c->contact['firstname'] . '</a>',
                    'mails' => $mail,
                    'phone' => (isset($add[0]) ? $add[0]['phone1'] : ''),
                    'class' => $class
                );
            }
        }
        
        $tGen = new TableGenerator('contact-list');
        
        $cont = '<table class="contact-list">';
        
        //$cont .= '<caption>' . $this->contactList->getGroupCaption() . '</caption>';
        
        if (count($data) > 0)
            if ($this->expand)
                $cont .= $tGen->generateBody($data,array('cont'),'class');
            else 
                $cont .= $tGen->generateBody($data,array('display_name','phone','mails'),'class');
        else
            $cont .= '<tr class="noentry"><td>No Entries.</td></tr>';
        
        $cont .= '</table>';
        
        return $cont;   
    }
}
?>
